<?php

namespace App\Services;

use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use App\Models\CreditCardCurrencyRate;
use App\Services\MasterCardConversionRates;
use App\Repositories\CurrencyRateRepository;

class CurrencyConverterService
{
    protected $currencyRateRepository;

    public function __construct(CurrencyRateRepository $currencyRateRepository)
    {
        $this->currencyRateRepository = $currencyRateRepository;
    }

    public function getConversionRate($baseCurrency, $quoteCurrency)
    {
        // 1. Check database for existing rate
        $rate = $this->currencyRateRepository->findRate($baseCurrency, $quoteCurrency);

        if ($rate) {
            return [
                'status' => 'success',
                'source' => 'database',
                'rate' => $rate->conversion_rate,
                'date' => $rate->quote_date,
            ];
        }

        // 2. If not found, fetch from Python script
        $conversionRate = $this->fetchFromPythonScript($baseCurrency, $quoteCurrency);

        if (!$conversionRate) {
            return [
                'status' => 'error',
                'message' => 'Unable to fetch conversion rate',
            ];
        }

        // 3. Save new rate in the database
        $this->currencyRateRepository->saveRate($baseCurrency, $quoteCurrency, $conversionRate, 'Visa');

        return [
            'status' => 'success',
            'source' => 'python_script',
            'rate' => $conversionRate,
            'date' => now()->toDateString()
        ];
    }

    private function fetchFromPythonScript($baseCurrency, $quoteCurrency)
    {
        $command = "python3 ExternalScript/send_request.py 1 $baseCurrency $quoteCurrency 2>&1";
        $output = shell_exec($command);

        Log::info("Python script output: " . print_r($output, true));

        if (!$output) {
            return null;
        }

        $decodedOutput = json_decode($output, true);

        if (json_last_error() !== JSON_ERROR_NONE || empty($decodedOutput['fxRateWithAdditionalFee'])) {
            return null;
        }

        return $decodedOutput['fxRateWithAdditionalFee'];
    }
}
